/**
  ******************************************************************************
  * @file    cp_init.c 
  * @author  Ruediger R. Asche
  * @version V1.0.0
  * @date    July 14, 2016
  * @brief   Initializes the network layers (Middleware and Application ends)
  ******************************************************************************
  * @attention
  *
  * THE PRESENT FIRMWARE WHICH IS FOR GUIDANCE ONLY AIMS AT PROVIDING CUSTOMERS
  * WITH CODING INFORMATION REGARDING THEIR PRODUCTS IN ORDER FOR THEM TO SAVE
  * TIME. AS A RESULT, THE AUTHOR SHALL NOT BE HELD LIABLE FOR ANY
  * DIRECT, INDIRECT OR CONSEQUENTIAL DAMAGES WITH RESPECT TO ANY CLAIMS ARISING
  * FROM THE CONTENT OF SUCH FIRMWARE AND/OR THE USE MADE BY CUSTOMERS OF THE
  * CODING INFORMATION CONTAINED HEREIN IN CONNECTION WITH THEIR PRODUCTS.
  ******************************************************************************  
  */ 

/******************************************************************************
 * Includes
 *****************************************************************************/

#include "project.h"

#include "cp.h"

extern void NetInit(void);  // Platform specific callback fn to set up the network. 


// relocate these!!!
#define CP_USERDEFINEDTCPPORT 7777
#define CP_USERDEFINEDREMOTETCPADDRESS ((192<<24)+(168<<16)+(2<<8)+20)
#ifdef SIMULATE_LOOPBACK
#define CP_LOCALLOOPBACK               ((127<<24)+(  0<<16)+(0<<8)+1)
#define CP_USERDEFINEDCOMMTARGET CP_LOCALLOOPBACK
#else
#define CP_USERDEFINEDCOMMTARGET CP_USERDEFINEDREMOTETCPADDRESS
#endif

#ifdef SIMULATE_LOOPBACK
// returns 0 if we simulate a dropped connection 
TickType_t g_TicksUntilSimConnLoss,g_TimeStampOfLastFordedDisconnect; 
unsigned long Simulate_ConnectionLoss(void)
{
    if (!g_TicksUntilSimConnLoss)
    {
        g_TicksUntilSimConnLoss = rand()%60000;
        g_TimeStampOfLastFordedDisconnect = xTaskGetTickCount();
    }
    if ((xTaskGetTickCount()-g_TimeStampOfLastFordedDisconnect) >= g_TicksUntilSimConnLoss)
    {
        g_TicksUntilSimConnLoss = 0;
        return 0;
    }
    return 1;
}
#endif


// The receiver task tries to receives peer data in chunks, reading as many characters as possible up to a max of MAXFASTRECVBUF.
// This should be chosen heuristically depending on the average number of characters in a payload packet. 

#define MAXFASTRECVBUF 40

/** @brief Task function to asynchronously process input over a connection.
 *
 *  @param p_CommHandle communication processor
 *
 *  @return none
 *
 *  Common to clients and servers. 
 */

void ReceiverTask(void *p_Arg)
{
    PCP_PROCESSOR a_cpP = (PCP_PROCESSOR)p_Arg;
    unsigned char *a_FastRecvBuf = (unsigned char *)pvPortMalloc(MAXFASTRECVBUF);
    if (!a_FastRecvBuf) STATIC_SANITY_CHECK;
    while (1)
    {
        // after each logical connection attempt, start out w/ unauthorized
        unsigned long a_StillInProgress = 1;
        a_cpP->m_AuthStatus &= ~CP_AUTHSTATUS_FLAG_AUTHRECEIVED;
        ulTaskNotifyTake( pdTRUE, portMAX_DELAY );
        CP_PrepareInBuf(&a_cpP->m_FSMStruct);
        a_cpP->m_FSMStruct.m_FSM = CP_FSM_IN_SCANNING;
        do
        {
            int a_ReadResult = recv(a_cpP->m_Socket,a_FastRecvBuf,MAXFASTRECVBUF,0);
#ifdef SIMULATE_LOOPBACK
            if ((a_ReadResult >= 1) && Simulate_ConnectionLoss())
#else
            if (a_ReadResult >= 1)
#endif
            {
                int aLoop;
                for (aLoop =0; aLoop< a_ReadResult; aLoop++)
                    CP_ProcessInChar(a_FastRecvBuf[aLoop],a_cpP);
            }
            else
            {
                int a_Error;
                socklen_t a_ReturnSize = sizeof(a_Error);
                getsockopt(a_cpP->m_Socket, SOL_SOCKET, SO_ERROR, (void *)&a_Error,&a_ReturnSize);
                if (a_Error != EWOULDBLOCK)
                {
                    CP_MSGPUMPINFO a_CurrentComm;
                    a_CurrentComm.m_Cmd = CP_CMD_TERMINATE_CONNECTION;
                    CP_SendPriQueue(a_cpP->m_CommQueue,CP_PRIQUEUE_CRITICALPRI,&a_CurrentComm);
                    a_StillInProgress = 0;
                }
            } // recv didn't get a character
        } while (a_StillInProgress);
    } // infinite task loop
    // never gets here, just for completeness's sake
    vPortFree(a_FastRecvBuf);
}

/** @brief Common code for server and client to initialize a communication. Sets up a processor and a corresponding receiver process.
 *
 *
 *  @return A pointer to a CP_PROCESSOR data structure or 0 in case of failure.
 *
 * local function called by both the client and server initialization routines.
 */

static PCP_PROCESSOR cp_Init(void)
{
    PCP_PROCESSOR a_CommInfo = (PCP_PROCESSOR) pvPortMalloc(sizeof(CP_PROCESSOR));
    if (a_CommInfo)
    {
        memset(a_CommInfo,0,sizeof(*a_CommInfo));
        CP_CreatePriQueue(&a_CommInfo->m_CommQueue);
        a_CommInfo->m_WriterMutex = xSemaphoreCreateMutex();
        if (a_CommInfo->m_CommQueue && a_CommInfo->m_WriterMutex)
        {
            if (xTaskCreate(ReceiverTask, "ReceiverTask", RECEIVERTASKSTACKSIZE,
			          (void *)a_CommInfo, RECEIVERTASKPRIORITY, &a_CommInfo->m_AssociatedReceiverTask) == pdTRUE)
            {
                return a_CommInfo;
            } // could create task
        }
    }
    if (a_CommInfo)
    {
        if (a_CommInfo->m_CommQueue) CP_DeletePriQueue(a_CommInfo->m_CommQueue);
        if (a_CommInfo->m_WriterMutex) vSemaphoreDelete(a_CommInfo->m_WriterMutex);
        vPortFree (a_CommInfo);
    }
    return 0;
}


#ifdef RUN_AS_SERVER  

/** @brief Task function to execute the server side of a communication
 *
 *  @param p_CommHandle communication processor
 *
 *  @return none
 *
 */

void RunCommLoopServer(void *p_Param)
{
    PCP_PROCESSOR a_CommInfo = (PCP_PROCESSOR)p_Param;
    // Some TCP/IP implementations allow one-time initialization of the stack before starting the operating system. Lwip
    // requires a running a running OS (unless it is configured to run completly without one), so we must start lwip in a
    // task. 
    NetInit();
    if (!a_CommInfo || (cp_CommLoopServer(CP_USERDEFINEDTCPPORT,a_CommInfo) == CP_STATUSCODE_FATAL))
    {
        STATIC_SANITY_CHECK;
    };
}

// The stack size is configuration dependent and may need to be adjusted. This one was determined in practice for the FRDM64K evalk board.

#define COMMSERVER_STACKSIZE ((configMINIMAL_STACK_SIZE*3)*1.5)

/** @brief Initializes the server side of a communication. Sets up a processor and a corresponding receiver process.
 *
 *
 *  @return A pointer to a CP_PROCESSOR data structure or 0 in case of failure.
 *
 */

CP_COMMHANDLE cp_initCommLayerServer(void)
{
    PCP_PROCESSOR a_CommInfo = cp_Init();    
    if (a_CommInfo)
    {
        a_CommInfo->m_AuthStatus |= CP_AUTHSTATUS_FLAG_PEERMUSTAUTHENTICATE; // only server is required to wait for auth by the peer!
        xTaskCreate(RunCommLoopServer, "CommServer", COMMSERVER_STACKSIZE,
            (void *)a_CommInfo, COMMTASKPRIORITY, (TaskHandle_t *) 0);
    }
    return a_CommInfo;
}

#endif

#ifdef RUN_AS_CLIENT

#define BROKENCONNBACKOFFTILE 200

/** @brief Task function to execute the client side of a communication
 *
 *  @param p_CommHandle communication processor
 *
 *  @return none
 *
 */

void RunCommLoopClient(void *p_Param)
{
    PCP_PROCESSOR a_CommInfo = (PCP_PROCESSOR)p_Param;
    // Some TCP/IP implementations allow one-time initialization of the stack before starting the operating system. Lwip
    // requires a running a running OS (unless it is configured to run completly without one), so we must start lwip in a
    // task. 
    NetInit();
    {
        if (a_CommInfo)
        {
            while (1)
            {
                if (cp_CommLoopClient(CP_USERDEFINEDCOMMTARGET,CP_USERDEFINEDTCPPORT,a_CommInfo) == CP_STATUSCODE_FATAL)
                    vTaskDelay(BROKENCONNBACKOFFTILE);   // to prevent an infinite CPU bound loop
            }
        }
    }
}

// The stack size is configuration dependent and may need to be adjusted. This one was determined in practice for the FRDM64K evalk board.

#define COMMCLIENT_STACKSIZE ((configMINIMAL_STACK_SIZE*3)*1.5)

/** @brief Initializes the server side of a communication. Sets up a processor and a corresponding receiver process.
 *
 *
 *  @return A pointer to a CP_PROCESSOR data structure or 0 in case of failure.
 *
 */

CP_COMMHANDLE cp_initCommLayerClient(void)
{
    PCP_PROCESSOR a_CommInfo = cp_Init();
    if (a_CommInfo)
    {
        xTaskCreate(RunCommLoopClient, "CommClient", COMMCLIENT_STACKSIZE,
            (void *)a_CommInfo, COMMTASKPRIORITY, (TaskHandle_t *) 0);
    }
    return a_CommInfo;
}
#endif
